#include <iostream>
#include <stdexcept>

// ---- Template Selection Helper
  
template<bool Condition, class T = void>
struct enable_if {
  typedef T type;
};

template<class T>
struct enable_if<false, T> {};

// ---- Meta Assertions

#define STATIC_ASSERT(x) static const int STATIC_ASSERT_ ## __COUNT__ [(x)]

// ---- Runtime Error

struct out_of_range : virtual std::out_of_range {
  out_of_range() : std::out_of_range("value out of range") {}
};

// ---- Our Actual 'restricted' type

template<class T, T Min, T Max>
struct restricted {
  STATIC_ASSERT((Min < Max));
  
  T value;
  
  explicit restricted(T o)
    : value(o) {
    if(value < Min || Max < value) { throw out_of_range(); }
  }
  
  template<T OMin, T OMax>
  restricted(restricted<T, OMin, OMax> o
    , typename enable_if< OMin <= Max && Min <= OMax >::type* = 0
  )
    : value(o.value) {
    if(value < Min || Max < value) { throw out_of_range(); }
  }
  
  template<T OMin, T OMax>
  restricted& operator=(
    typename enable_if< OMin <= Max && Min <= OMax,
      restricted<int, OMin, OMax> >::type) {
    if(o.value < Min || Max < o.value) { throw out_of_range(); }
    value = o.value;
  }
  
  //
  // For operator+ we just have to widen the potential
  // range.  No magic here.
  //
  template<T OMin, T OMax>
  restricted<T, Min + OMin, Max + OMax> operator+(restricted<T, OMin, OMax> o) {
    return restricted<T, Min + OMin, Max + OMax>(value + o.value);
  }
};

// ---- How to use it

int main() {
  restricted<int, 0, 10> i(5);
  restricted<int, -10, 10> j = i + i; // results in a restricted<int,0,20>
  //
  // TODO: Try:
  //restricted<int, -10, -5> j = i;
}

// META Section
//
// With the exception of enable_if, not much of template
// _metaprogramming_ has happened yet.  Well, even for
// such a simple class as this, we already get to a problem,
// if somebody for instance would like to use some custom
// numeric class wrapper (i.e. a Rational or BigNum class).
//
// Expressions like Min + OMin would not work.
// So...
